<?php
declare(strict_types=1);

class Router
{
    private array $routes = [];

    public function get(string $pattern, callable|array $handler): void
    {
        $this->addRoute('GET', $pattern, $handler);
    }

    public function post(string $pattern, callable|array $handler): void
    {
        $this->addRoute('POST', $pattern, $handler);
    }

    public function put(string $pattern, callable|array $handler): void
    {
        $this->addRoute('PUT', $pattern, $handler);
    }

    public function patch(string $pattern, callable|array $handler): void
    {
        $this->addRoute('PATCH', $pattern, $handler);
    }

    public function delete(string $pattern, callable|array $handler): void
    {
        $this->addRoute('DELETE', $pattern, $handler);
    }

    private function addRoute(string $method, string $pattern, callable|array $handler): void
    {
        // Convert :param to named regex groups
        $regex = preg_replace('/\/:([a-zA-Z_]+)/', '/(?P<$1>[^/]+)', $pattern);
        $regex = '#^' . $regex . '$#';
        $this->routes[] = compact('method', 'pattern', 'regex', 'handler');
    }

    public function dispatch(): void
    {
        $method = $_SERVER['REQUEST_METHOD'];
        $uri    = parse_url($_SERVER['REQUEST_URI'], PHP_URL_PATH);

        // ── Strip subdirectory prefix (penting untuk Laragon www/) ─────────────
        // SCRIPT_NAME = /backend-php/index.php
        // Kita ambil dirname-nya = /backend-php
        $scriptDir = rtrim(dirname($_SERVER['SCRIPT_NAME']), '/');

        // Hapus prefix subdirectory dari URI
        // /backend-php/api/routes  →  /api/routes
        if ($scriptDir !== '' && $scriptDir !== '/' && str_starts_with($uri, $scriptDir)) {
            $uri = substr($uri, strlen($scriptDir));
        }

        // Normalisasi: pastikan selalu diawali /
        $uri = '/' . ltrim($uri, '/');

        // Hapus trailing slash kecuali root
        if ($uri !== '/') {
            $uri = rtrim($uri, '/');
        }

        foreach ($this->routes as $route) {
            if ($route['method'] !== $method) {
                continue;
            }

            if (preg_match($route['regex'], $uri, $matches)) {
                // Ambil hanya named params (string key)
                $params = array_filter($matches, 'is_string', ARRAY_FILTER_USE_KEY);

                $handler = $route['handler'];

                // Closure
                if ($handler instanceof Closure) {
                    $handler($params);
                    return;
                }

                // [ClassName, methodName]
                [$class, $action] = $handler;
                require_once __DIR__ . "/../controllers/{$class}.php";
                $controller = new $class();
                $controller->$action($params);
                return;
            }
        }

        // 404
        http_response_code(404);
        echo json_encode(['message' => 'Endpoint not found.', 'uri' => $uri]);
    }
}
